///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//  Copyright  NetworkDLS 2002, All rights reserved
//
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF 
// ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO 
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A 
// PARTICULAR PURPOSE.
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef _SQLIMPORT_CPP
#define _SQLIMPORT_CPP
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include <Windows.H>
#include <WindowsX.H>
#include <ShellAPI.H>
#include <Stdio.H>
#include <Stdlib.H>
#include <SQL.H>
#include <SQLExt.H>

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include "../Resources/Resource.H"

#include "../../SharedClasses/SQLClass/cSQL.H"
#include "../../SharedClasses/SQLClass/cRecordSet.H"
#include "../../SharedClasses/CGetPKs/CGetPKs.H"
#include "../../SharedClasses/CMemPool/CMemPool.H"
#include "../../SharedClasses/CDynString/CDynString.H"
#include "../CSockSrvr/CSockSrvr.H"

#include "../../SharedSource/Debug.H"
#include "../../SharedSource/NSWFL.H"
#include "../../SharedSource/Common.H"
#include "../../SharedSource/CRC32.H"

#include "Init.H"
#include "Routines.H"
#include "Command.H"
#include "Console.H"
#include "SQLImport.H"

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void FreeSingleRowDataColumns(struct _SQLImportMembers *SIM, int iColumnCount)
{
    Assert(SIM == NULL, "Proc: FreeSingleRowDataColumns");
	
	int iColumnPos = 0;
    while(iColumnPos < iColumnCount)
    {
		gMem.Free(SIM->SingleRowData[iColumnPos]);
        SIM->SingleRowData[iColumnPos] = NULL;
        iColumnPos++;
    }

	SIM->iOutDataType[iColumnPos] = 0;
	SIM->iOutColumnSize[iColumnPos] = 0;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void FreeColumnNames(struct _SQLImportMembers *SIM, int iColumnCount)
{
	Assert(SIM == NULL, "Proc: FreeColumnNames");

	int iColumnPos = 0;
    while(iColumnPos < iColumnCount)
    {
        gMem.Free(SIM->ColumnNames[iColumnPos]);
        SIM->ColumnNames[iColumnPos] = NULL;
        iColumnPos++;
    }

	gMem.Free(SIM->iOutDataType);
	gMem.Free(SIM->iOutColumnSize);
	
	gMem.Free(SIM->ColumnNames);
    SIM->ColumnNames = NULL;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void FreeSingleRowData(struct _SQLImportMembers *SIM)
{
    Assert(SIM == NULL, "Proc: FreeSingleRowData");

	gMem.Free(SIM->SingleRowData);
    SIM->SingleRowData = NULL;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool ImportSQLResults(CSockSrvr *pSockSrvr, int iClient, int *iOutID, bool bCheckExistance, char *InFileName)
{
	Assert(pSockSrvr == NULL, "Proc: ImportSQLResults");

	char sSQL[2048];
	char sImportDB[255];                  //The name of the database that were going to be importing into.
	char sImportTable[255];               //The name of the table that were going to be importing into.
    char sStatusText[MAX_STATUS_TEXT];    //Just some RAM for status text and error messages.
    char sAltTemp[MAX_STATUS_TEXT];       //Just some RAM for status text and error messages.

    int *iUniuqeKeyCols = NULL;           // Contains the Column position of the uniuqe Columns.
    int iUniuqeKeysNeeded = 0;            // The ammount of uniuqe keys we have to find.
    int iUniuqeKeysFound = 0;             // The ammount of uniuqe keys we have found.
    int iTotalLengthOfAllColumnNames = 0; // For better memory management.
    int iTotalLengthOfAllRowData = 0;     // For better memory management.
    int iTemp = 0;
    int iColumnCount = 0;
    int iRowCount = 0;
    int iColNameLen = 0;
    int iColumnPos = 0;
	int iID = 0;
	bool bIsDelete = false; //Is the data we received a list of rows to delete?

	CGetPKs cMyPKs;

    FILE *hSource = NULL;

    _SQLImportMembers SIM;

    CRecordSet rsTemp;

    if(fopen_s(&hSource, InFileName, "rb") != 0)
    {
		sprintf_s(sStatusText, sizeof(sStatusText),
			"Error in ImportSQLResults:: TargetFile open error. Name: \"%s\"", InFileName);
		WriteLogEx(pSockSrvr->icClientID[iClient], sStatusText, EVENT_ERROR);
        return false;
    }

    fread(&iTemp, sizeof(iTemp), 1, hSource); // Read import database name len.
    fread(sImportDB, sizeof(char), iTemp, hSource); // Read database table name.
    sImportDB[iTemp] = '\0';

	strcpy_s(sAltTemp, sizeof(sAltTemp), sImportDB);

	strcpy_s(sImportDB, sizeof(sImportDB), "SQLExch_");
	strcat_s(sImportDB, sizeof(sImportDB), CCI[iClient].sCompanyDBPrefix);
	strcat_s(sImportDB, sizeof(sImportDB), "_");
	strcat_s(sImportDB, sizeof(sImportDB), sAltTemp);

	sprintf_s(sSQL, sizeof(sSQL), "USE [%s]", sImportDB);
	if(!CCI[iClient].cCustSQL.ExecuteNonQuery(sSQL))
	{
		WriteLogEx(pSockSrvr->icClientID[iClient], "Failed to change client database context. Performance will be affected.", EVENT_WARN);
	}

	fread(&iTemp, sizeof(iTemp), 1, hSource); // Read import table name len.
    fread(sImportTable, sizeof(char), iTemp, hSource); // Read import table name.
    sImportTable[iTemp] = '\0';

	fread(&iColumnCount, sizeof(iColumnCount), 1, hSource); // Read number of Columns.
    fread(&iRowCount, sizeof(iRowCount), 1, hSource);       // Read number of rows.
    fread(&iID, sizeof(iID), 1, hSource);                   // Read the row ID of the associated Statments row.

	*iOutID = iID;

	sprintf_s(sStatusText, sizeof(sStatusText),
		"Processing transaction data. [%s.%s - %d Rows, %d Columns]", 
		sImportDB, sImportTable, iRowCount, iColumnCount);
	WriteLogEx(pSockSrvr->icClientID[iClient], sStatusText, EVENT_NONE);

	char sRight[100];
	Right(sImportTable, sRight, iTemp, 15);

	if( strcmp(sRight, "_SQLExch_Delete") == 0)
	{
		bIsDelete = true;
		
		//Turncate the import table. Trimming off the "_SQLExch_Delete"
		sImportTable[iTemp - 15] = '\0';
	}

	//Get a list of primary keys for the import table.
	cMyPKs.Get(&CCI[iClient].cCustSQL, sImportDB, "DBO", sImportTable);

	iUniuqeKeysNeeded = cMyPKs.iPKs;

	if(iUniuqeKeysNeeded > 0)
	{
		iUniuqeKeyCols = (int *) gMem.Allocate(sizeof(int), iUniuqeKeysNeeded);
		Assert(!iUniuqeKeyCols, "Memory Allocation Error.");
	}
	else{
		//No primary keys, cannot import this data.
		WriteLogEx(pSockSrvr->icClientID[iClient], "Error in ImportSQLResults:: Could not obtain a list of primary keys.", EVENT_ERROR);

        cMyPKs.Free();
		fclose(hSource);
        return false;
	}

	//Allocate RAM for the list of column names.
	SIM.ColumnNames = (char **) gMem.Allocate(sizeof(char *), iColumnCount + 1);
	Assert(!SIM.ColumnNames, "Memory Allocation Error.");
	SIM.iOutDataType = (int *) gMem.Allocate(sizeof(int), iColumnCount + 1);
	Assert(!SIM.iOutDataType, "Memory Allocation Error.");
	SIM.iOutColumnSize = (int *) gMem.Allocate(sizeof(int), iColumnCount + 1);
	Assert(!SIM.iOutColumnSize, "Memory Allocation Error.");

	//Loop through, reading all of the column names from the file.
	while(iColumnPos < iColumnCount)
    {
        fread(&SIM.iOutDataType[iColumnPos], sizeof(int), 1, hSource);
        fread(&SIM.iOutColumnSize[iColumnPos], sizeof(int), 1, hSource);

		fread(&iColNameLen, sizeof(iColNameLen), 1, hSource);
        SIM.ColumnNames[iColumnPos] = (char *) gMem.Allocate(sizeof(char), iColNameLen + 1);
		Assert(!SIM.ColumnNames[iColumnPos], "Memory Allocation Error.");
        fread(SIM.ColumnNames[iColumnPos], sizeof(char), iColNameLen, hSource);
        SIM.ColumnNames[iColumnPos][iColNameLen] = '\0';
        iTotalLengthOfAllColumnNames = (iTotalLengthOfAllColumnNames + iColNameLen);

        //If this column name matches one of our Primary Keys then record its position
		//	in iUniuqeKeyCols[]
        if(iUniuqeKeysFound < iUniuqeKeysNeeded)
        {
            int iUniuqeKeyCount = 0;
            while(iUniuqeKeyCount < iUniuqeKeysNeeded)
            {
				if(_strcmpi(cMyPKs.sPKs[iUniuqeKeyCount], SIM.ColumnNames[iColumnPos]) == 0)
                {
                    iUniuqeKeyCols[iUniuqeKeyCount] = iColumnPos;
                    iUniuqeKeysFound++;
                }
                iUniuqeKeyCount++;
            }
        }

        iColumnPos++;
    }

	int iUniuqeKeyCount = 0;
    while(iUniuqeKeysNeeded < iUniuqeKeysFound)
    {
        sprintf_s(sStatusText, sizeof(sStatusText),
			"Error in ImportSQLResults:: Required uniuqe keys were not found for: '%s'.", sImportTable);
		WriteLogEx(pSockSrvr->icClientID[iClient], sStatusText, EVENT_ERROR);

        FreeColumnNames(&SIM, iColumnCount);

        cMyPKs.Free();
        fclose(hSource);
        return false;
    }

    if( pSockSrvr->bcConnected[iClient] == false )
    {
		if(bIsDelete)
		{
			WriteLogEx(pSockSrvr->icClientID[iClient], "Disconnected before data import.", EVENT_WARN);
		}
		else{
			WriteLogEx(pSockSrvr->icClientID[iClient], "Disconnected before data deletion.", EVENT_WARN);
		}

        cMyPKs.Free();
        fclose(hSource);
        return false;
    }

    //Allocate an Array large enough to hold all of the row data.
	SIM.SingleRowData = (char **) gMem.Allocate(sizeof(char *), iColumnCount + 1);
	Assert(!SIM.SingleRowData, "Memory Allocation Error.");

	if(CCI[iClient].bRequestInit)
	{
		sprintf_s(sSQL, sizeof(sSQL),
			"DELETE FROM [%s].[%s].[%s]",
			sImportDB, "DBO", sImportTable);
		CCI[iClient].cCustSQL.ExecuteNonQuery(sSQL);
	}

	CDynString sSrvSQL;

	if(!sSrvSQL.Init())
	{
		WriteLogEx(pSockSrvr->icClientID[iClient], "Failed to allocate memory for sSrvSQL.", EVENT_ERROR);

		FreeSingleRowDataColumns(&SIM, iColumnCount);
		FreeSingleRowData(&SIM);
		FreeColumnNames(&SIM, iColumnCount);

	    cMyPKs.Free();
		fclose(hSource);
		return false;
	}

	int iRowDataLen = 0;
    int iRowPos = 0;

    while(iRowPos < iRowCount)
    {
        int iBaseMemoryRequirement = 0;
        iTotalLengthOfAllRowData = 0;

        // Get row data.
        iColumnPos = 0;
        while(iColumnPos < iColumnCount)
        {
            fread(&iRowDataLen, sizeof(iRowDataLen), 1, hSource);
            if((SIM.SingleRowData[iColumnPos] = (char *) gMem.Allocate(sizeof(char), iRowDataLen + 1)) == NULL)
			{
				WriteLogEx(pSockSrvr->icClientID[iClient], "Failed to allocate memory for SingleRowData[].", EVENT_ERROR);

				Assert(!SIM.SingleRowData[iColumnPos], "Memory Allocation Error.");

				cMyPKs.Free();
		        fclose(hSource);
				return false;
			}
			fread(SIM.SingleRowData[iColumnPos], sizeof(char), iRowDataLen, hSource);
			
            SIM.SingleRowData[iColumnPos][iRowDataLen] = '\0';
            iTotalLengthOfAllRowData = (iTotalLengthOfAllRowData + iRowDataLen);
            iColumnPos++;
        }

        iBaseMemoryRequirement = ((iTotalLengthOfAllColumnNames + iTotalLengthOfAllRowData) + (iColumnCount * 5)) + 1024;

        char sWhereStatement[1024];
        char sCheckExistanceStatement[1024];
        int iCountOfRowsThatAlreadyExist = 0;

        // Assemble the where statement from the row data, the uniuqe key names and positions.
        if(iUniuqeKeysFound > 0)
        {
            // Assemble the where statement
            sprintf_s(sWhereStatement, sizeof(sWhereStatement),
				"WHERE %s = %s", cMyPKs.sPKs[0], SIM.SingleRowData[iUniuqeKeyCols[0]]);
        	int iUniuqeKeyCount = 1;
            while(iUniuqeKeyCount < iUniuqeKeysFound)
            {
                strcat_s(sWhereStatement, sizeof(sWhereStatement), " AND [");
                strcat_s(sWhereStatement, sizeof(sWhereStatement), cMyPKs.sPKs[iUniuqeKeyCount]);
                strcat_s(sWhereStatement, sizeof(sWhereStatement), "] = ");
                strcat_s(sWhereStatement, sizeof(sWhereStatement), SIM.SingleRowData[iUniuqeKeyCols[iUniuqeKeyCount]]);
                iUniuqeKeyCount++;
            }

            // Assemble the check existance statement
            sprintf_s(sCheckExistanceStatement, sizeof(sCheckExistanceStatement), "SELECT %s", cMyPKs.sPKs[0]);
            strcat_s(sCheckExistanceStatement, sizeof(sCheckExistanceStatement), " FROM [");
            strcat_s(sCheckExistanceStatement, sizeof(sCheckExistanceStatement), sImportDB);
            strcat_s(sCheckExistanceStatement, sizeof(sCheckExistanceStatement), "].[DBO].[");
            strcat_s(sCheckExistanceStatement, sizeof(sCheckExistanceStatement), sImportTable);
            strcat_s(sCheckExistanceStatement, sizeof(sCheckExistanceStatement), "] ");
            strcat_s(sCheckExistanceStatement, sizeof(sCheckExistanceStatement), sWhereStatement);
        }

		if(bIsDelete == false && bCheckExistance)
		{
			if(iUniuqeKeysFound > 0)
			{
				if(!CCI[iClient].cCustSQL.Execute(sCheckExistanceStatement, &rsTemp))
				{
					WriteLogEx(pSockSrvr->icClientID[iClient], "Error in ImportSQLResults:: Execute failed on sCheckExistanceStatement. (See log file for more information)", EVENT_ERROR);

					sprintf_s(sStatusText, sizeof(sStatusText), "Statement: %s", sCheckExistanceStatement);
					WriteLogEx(pSockSrvr->icClientID[iClient], sStatusText, EVENT_NONE);

					//Need to do some advanced error reporting.
					if(CCI[iClient].cCustSQL.GetErrorMessage(sAltTemp, sizeof(sAltTemp), rsTemp.hSTMT))
					{
						sprintf_s(sStatusText, sizeof(sStatusText), "Error Msg: %s", sAltTemp);
						WriteLogEx(pSockSrvr->icClientID[iClient], sStatusText, EVENT_NONE);
					}

					FreeSingleRowDataColumns(&SIM, iColumnCount);
					FreeSingleRowData(&SIM);
					FreeColumnNames(&SIM, iColumnCount);

			        cMyPKs.Free();
					fclose(hSource);
					return false;
				}

				iCountOfRowsThatAlreadyExist = rsTemp.RowCount;
				rsTemp.Close();
			}
		}

        //----------------------------------------------------------------------------------------------
        if(bIsDelete)
		{
			sSrvSQL.Append("DELETE FROM [");
			sSrvSQL.Append(sImportDB);
			sSrvSQL.Append("].[DBO].[");
			sSrvSQL.Append(sImportTable);
			sSrvSQL.Append("] ");
			sSrvSQL.Append(sWhereStatement);
		}
		else if(iCountOfRowsThatAlreadyExist == 0) // Do insert
        {
			sSrvSQL.Append("INSERT INTO [");
			sSrvSQL.Append(sImportDB);
			sSrvSQL.Append("].[DBO].[");
			sSrvSQL.Append(sImportTable);
			sSrvSQL.Append("] (");

            iColumnPos = 0;
            while(iColumnPos < iColumnCount)
            {
                sSrvSQL.Append("[");
                sSrvSQL.Append(SIM.ColumnNames[iColumnPos]);
                sSrvSQL.Append("]");
                if(iColumnPos < iColumnCount - 1)
                {
                    sSrvSQL.Append(",");
                }
                iColumnPos++;
            }
            sSrvSQL.Append(") VALUES(");

            iColumnPos = 0;
            while(iColumnPos < iColumnCount)
            {
				sSrvSQL.Append(SIM.SingleRowData[iColumnPos]);
                if(iColumnPos < iColumnCount - 1)
                {
                    sSrvSQL.Append(",");
                }
                iColumnPos++;
            }
            sSrvSQL.Append(") ");
        }
        //----------------------------------------------------------------------------------------------
        else { // Do update

			sSrvSQL.Append("UPDATE [");
			sSrvSQL.Append(sImportDB);
			sSrvSQL.Append("].[DBO].[");
			sSrvSQL.Append(sImportTable);
			sSrvSQL.Append("] SET");

            iColumnPos = 0;
            while(iColumnPos < iColumnCount)
            {
				sSrvSQL.Append("[");
                sSrvSQL.Append(SIM.ColumnNames[iColumnPos]);
                sSrvSQL.Append("]=");
                sSrvSQL.Append(SIM.SingleRowData[iColumnPos]);
                if(iColumnPos < iColumnCount - 1)
                {
                    sSrvSQL.Append(",");
                }
                iColumnPos++;
            }

            sSrvSQL.Append(" ");
            sSrvSQL.Append(sWhereStatement);
        }

        //----------------------------------------------------------------------------------------------

        if(sSrvSQL.ciBufSz > 0)
        {
			sSrvSQL.AppendEx("\0", 1);

			if(!CCI[iClient].cCustSQL.Execute(sSrvSQL.csDataBuf, &rsTemp))
            {
				WriteLogEx(pSockSrvr->icClientID[iClient],
					"Error in ImportSQLResults:: Execute failed on sSrvSQL. (See log file for more information)", EVENT_ERROR);

				if(sSrvSQL.ciBufSz > (sizeof(sStatusText) - 25))
				{
					strcpy_s(sStatusText, sizeof(sStatusText), "Statement: ");
					strncat_s(sStatusText, sizeof(sStatusText), sSrvSQL.csDataBuf, (sizeof(sStatusText) - 25));
				}
				else {
					sprintf_s(sStatusText, sizeof(sStatusText), "Statement: %s", sSrvSQL.csDataBuf);
				}
				WriteLogEx(pSockSrvr->icClientID[iClient], sStatusText, EVENT_NONE);

				//Need to do some advanced error reporting.
				if(CCI[iClient].cCustSQL.GetErrorMessage(sAltTemp, sizeof(sAltTemp), rsTemp.hSTMT))
				{
					sprintf_s(sStatusText, sizeof(sStatusText), "Error Msg: %s", sAltTemp);
					WriteLogEx(pSockSrvr->icClientID[iClient], sStatusText, EVENT_NONE);
				}

				FreeSingleRowDataColumns(&SIM, iColumnCount);
                FreeSingleRowData(&SIM);
                FreeColumnNames(&SIM, iColumnCount);

                sSrvSQL.Free();

		        cMyPKs.Free();
                fclose(hSource);
                return false;
            }

            rsTemp.Close();
        }

		sSrvSQL.Truncate();

        FreeSingleRowDataColumns(&SIM, iColumnCount);
        //FreeSingleRowData(&SIM); // This memory needs to stay intact untill we are done with the import.

        iColumnPos++;

        iRowPos++;

        if( pSockSrvr->bcConnected[iClient] == false )
        {
		    sprintf_s(sStatusText, sizeof(sStatusText),
				"Disconnected durring data import, after %d successful rows.", iRowPos);
			WriteLogEx(pSockSrvr->icClientID[iClient], sStatusText, EVENT_WARN);

            FreeSingleRowData(&SIM);
            FreeColumnNames(&SIM, iColumnCount);

	        cMyPKs.Free();
            fclose(hSource);
            return false;
        }

		EnterCriticalSection(&csTransPerSecond);
		giTransProcessed++;
		LeaveCriticalSection(&csTransPerSecond);
    }

    sSrvSQL.Free();

	if(bIsDelete)
	{
		sprintf_s(sStatusText, sizeof(sStatusText),
			"Successfully deleted %d client data rows.", iRowPos);
	}
	else {
		sprintf_s(sStatusText, sizeof(sStatusText),
			"Successfully imported %d client data rows.", iRowPos);
	}
	WriteLogEx(pSockSrvr->icClientID[iClient], sStatusText, EVENT_NONE);

	//free some resources.
	gMem.Free(iUniuqeKeyCols);
	FreeSingleRowData(&SIM);
    FreeColumnNames(&SIM, iColumnCount); 
    cMyPKs.Free();
    fclose(hSource);

    return true;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#endif
